<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use App\Models\UserPlan;
use App\Models\Transaction;
use App\Enum\UserPlanStatus;
use App\Enum\TransactionType;
use App\Enum\TransactionMethod;
use App\Enum\TransactionStatus;
use Illuminate\Console\Command;
use App\Enum\TransactionDirection;
use Illuminate\Support\Facades\DB;

class ProcessDailyProfit extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:process-daily-profit';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process daily profit for active trading plans';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $today = Carbon::today();

        $userPlans = UserPlan::where('status', UserPlanStatus::ACTIVE)
            ->whereDate('starts_at', '<=', $today)
            ->get();

        foreach ($userPlans as $userPlan) {
            // 1. Stop if plan has ended
            if ($today->greaterThan($userPlan->ends_at)) {
                $this->completePlan($userPlan);
                continue;
            }

            // 2. Prevent double profit for same day
            $alreadyCredited = Transaction::where('user_id', $userPlan->user_id)
                ->where('type', TransactionType::PROFIT)
                ->whereDate('created_at', $today)
                ->where('description', 'LIKE', '%Plan #' . $userPlan->plan->name . '%')
                ->exists();

            if ($alreadyCredited) {
                continue;
            }

            // 3. Calculate daily profit
            $profit = ($userPlan->amount * $userPlan->daily_profit) / 100;

            DB::transaction(function () use ($userPlan, $profit) {

                // Credit user
                $user = $userPlan->user;
                $user->balance += $profit;
                $user->save();

                // Log transaction
                Transaction::create([
                    'user_id' => $userPlan->user_id,
                    'type' => TransactionType::PROFIT,
                    'direction' => TransactionDirection::CREDIT,
                    'description' => 'Daily profit from Plan #' . $userPlan->plan->name,
                    'amount' => $profit,
                    'method' => TransactionMethod::TRADE,
                    'transaction_at' => now(),
                    'reference_id' => generateReferenceId(),
                    'status' => TransactionStatus::COMPLETED
                ]);
            });
        }

        return Command::SUCCESS;
    }

    protected function completePlan(UserPlan $userPlan)
    {
        DB::transaction(function () use ($userPlan) {

            $user = $userPlan->user;

            $user->balance += $userPlan->amount;
            $user->save();

            // Update plan status
            $userPlan->status = UserPlanStatus::COMPLETED->value;
            $userPlan->save();

            // Log transaction
            Transaction::create([
                'user_id' => $userPlan->user_id,
                'type' => TransactionType::PROFIT,
                'direction' => TransactionDirection::CREDIT,
                'description' => 'Capital released from completed plan #' . $userPlan->plan->name,
                'amount' => $userPlan->amount,
                'method' => TransactionMethod::TRADE,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'status' => TransactionStatus::COMPLETED
            ]);
        });
    }
}
