<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\Market;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreMarketRequest;
use App\Http\Requests\UpdateMarketRequest;
use Illuminate\Support\Facades\Log;

class MarketController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Markets', 'active' => true]
        ];

        $markets = Market::latest()->get();

        $data = [
            'title' => 'Markets',
            'breadcrumbs' => $breadcrumbs,
            'markets' => $markets
        ];

        return view('dashboard.admin.market.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Markets', 'url' => route('admin.market.index')],
            ['label' => 'Create Market', 'active' => true]
        ];

        $data = [
            'title' => 'Create Market',
            'breadcrumbs' => $breadcrumbs,
        ];

        return view('dashboard.admin.market.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreMarketRequest $request)
    {
        $validated = $request->validated();

        try {
            DB::beginTransaction();

            $market = Market::create($validated);

            DB::commit();

            return redirect()->route('admin.market.show', $market->uuid);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Markets', 'url' => route('admin.market.index')],
            ['label' => 'Market Details', 'active' => true]
        ];

        $market = Market::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Market Details',
            'breadcrumbs' => $breadcrumbs,
            'market' => $market
        ];

        return view('dashboard.admin.market.show', $data);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Markets', 'url' => route('admin.market.index')],
            ['label' => 'Edit Market', 'active' => true]
        ];

        $market = Market::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Edit Market',
            'breadcrumbs' => $breadcrumbs,
            'market' => $market
        ];

        return view('dashboard.admin.market.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateMarketRequest $request, string $uuid)
    {
        $validated = $request->validated();

        try {
            DB::beginTransaction();

            $market = Market::where('uuid', $uuid)->firstOrFail();
            $market->update($validated);

            DB::commit();

            return redirect()->route('admin.market.show', $market->uuid);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function delete(string $uuid)
    {
        try {
            DB::beginTransaction();

            $market = Market::where('uuid', $uuid)->firstOrFail();
            $market->delete();

            DB::commit();

            return redirect()->route('admin.market.index')->with('success', 'Market deleted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }
}
