<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\Newsletter;
use Illuminate\Http\Request;
use App\Jobs\SendNewsletterJob;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\NewsletterSubscriber;
use Illuminate\Support\Facades\Log;

class NewsletterSubscriberController extends Controller
{
    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Newsletter Subscribers', 'active' => true]
        ];

        $newsletterSubscribers = NewsletterSubscriber::latest()->get();

        $data = [
            'title' => 'Newsletter Subscribers',
            'breadcrumbs' => $breadcrumbs,
            'newsletterSubscribers' => $newsletterSubscribers,
        ];

        return view('dashboard.admin.newsletter_subscriber.index', $data);
    }

    public function create()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Newsletter Subscribers', 'url' => route('admin.newsletter.subscriber.index')],
            ['label' => 'Send Newsletter', 'active' => true]
        ];

        $data = [
            'title' => 'Send Newsletter',
            'breadcrumbs' => $breadcrumbs,
        ];

        return view('dashboard.admin.newsletter_subscriber.create', $data);
    }

    public function store(Request $request)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'content' => 'required|string',
        ]);

        try {
            DB::beginTransaction();

            $newsletter = Newsletter::create([
                'subject' => $request->subject,
                'content' => $request->content,
            ]);

            dispatch(new SendNewsletterJob($newsletter));

            DB::commit();

            return redirect()->route('admin.newsletter.subscriber.index')->with('success', 'Newsletter sent successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('admin.newsletter.subscriber.index')->with('error', 'Failed to send newsletter.');
        }
    }

    public function delete(NewsletterSubscriber $subscriber)
    {
        try {
            DB::beginTransaction();

            $subscriber->delete();

            DB::commit();

            return redirect()->route('admin.newsletter.subscriber.index')->with('success', 'Subscriber deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->route('admin.newsletter.subscriber.index')->with('error', config('app.messages.error'));
        }
    }

    public function deleteAll()
    {
        try {
            DB::beginTransaction();

            NewsletterSubscriber::query()->delete();

            DB::commit();

            return redirect()->route('admin.newsletter.subscriber.index')->with('success', 'All subscribers deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->route('admin.newsletter.subscriber.index')->with('error', config('app.messages.error'));
        }
    }
}
