<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\Market;
use App\Models\Signal;
use App\Enum\SignalType;
use App\Enum\SignalAction;
use App\Enum\SignalDuration;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreSignalRequest;
use App\Http\Requests\UpdateSignalRequest;

class SignalController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Signals', 'active' => true]
        ];

        $signals = Signal::with(['market'])->latest()->get();

        $data = [
            'title' => 'Signals',
            'breadcrumbs' => $breadcrumbs,
            'signals' => $signals,
        ];

        return view('dashboard.admin.signal.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Signals', 'url' => route('admin.signal.index')],
            ['label' => 'Create Signal', 'active' => true]
        ];

        $markets = Market::latest()->get();

        $signalDurations = SignalDuration::cases();
        $signalTypes = SignalType::cases();
        $signalActions = SignalAction::cases();

        $data = [
            'title' => 'Create Signal',
            'breadcrumbs' => $breadcrumbs,
            'markets' => $markets,
            'signalDurations' => $signalDurations,
            'signalTypes' => $signalTypes,
            'signalActions' => $signalActions
        ];

        return view('dashboard.admin.signal.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreSignalRequest $request)
    {
        $validated = $request->validated();

        try {
            DB::beginTransaction();

            Signal::create($validated);

            DB::commit();

            return redirect()->route('admin.signal.index')->with('success', 'Signal created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Signals', 'url' => route('admin.signal.index')],
            ['label' => 'Signal Details', 'active' => true]
        ];

        $signal = Signal::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Signal Details',
            'breadcrumbs' => $breadcrumbs,
            'signal' => $signal
        ];

        return view('dashboard.admin.signal.show', $data);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Signals', 'url' => route('admin.signal.index')],
            ['label' => 'Edit Signal', 'active' => true]
        ];

        $markets = Market::latest()->get();

        $signalDurations = SignalDuration::cases();
        $signalTypes = SignalType::cases();
        $signalActions = SignalAction::cases();

        $signal = Signal::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Edit Signal',
            'breadcrumbs' => $breadcrumbs,
            'markets' => $markets,
            'signalDurations' => $signalDurations,
            'signalTypes' => $signalTypes,
            'signalActions' => $signalActions,
            'signal' => $signal
        ];

        return view('dashboard.admin.signal.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateSignalRequest $request, string $uuid)
    {
        $validated = $request->validated();

        try {
            DB::beginTransaction();

            Signal::where('uuid', $uuid)->update($validated);

            DB::commit();

            return redirect()->route('admin.signal.index')->with('success', 'Signal updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function delete(string $uuid)
    {
        try {
            DB::beginTransaction();

            Signal::where('uuid', $uuid)->delete();

            DB::commit();

            return redirect()->route('admin.signal.index')->with('success', 'Signal deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }
}
