<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Enum\Currency;
use App\Models\TradeBot;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreTradeBotRequest;
use App\Http\Requests\UpdateTradeBotRequest;
use Illuminate\Support\Facades\Log;

class TradeBotController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Trade Bots', 'active' => true]
        ];

        $tradeBots = TradeBot::latest()->get();

        $data = [
            'title' => 'Trade Bots',
            'breadcrumbs' => $breadcrumbs,
            'tradeBots' => $tradeBots,
        ];

        return view('dashboard.admin.trade_bot.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Trade Bots', 'url' => route('admin.trade.bot.index')],
            ['label' => 'Create Trade Bot', 'active' => true]
        ];

        $currencies = Currency::cases();

        $data = [
            'title' => 'Create Trade Bot',
            'breadcrumbs' => $breadcrumbs,
            'currencies' => $currencies
        ];

        return view('dashboard.admin.trade_bot.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreTradeBotRequest $request)
    {
        try {
            DB::transaction(function () use ($request) {
                TradeBot::create($request->validated());
            });

            return redirect()
                ->route('admin.trade.bot.index')
                ->with('success', 'Trade Bot created successfully');
        } catch (\Exception $e) {
            Log::error($e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'Failed to create trade bot');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Trade Bots', 'url' => route('admin.trade.bot.index')],
            ['label' => 'Trade Bot Details', 'active' => true]
        ];

        $tradeBot = TradeBot::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Trade Bot Details',
            'breadcrumbs' => $breadcrumbs,
            'tradeBot' => $tradeBot,
        ];

        return view('dashboard.admin.trade_bot.show', $data);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Trade Bots', 'url' => route('admin.trade.bot.index')],
            ['label' => 'Edit Trade Bot', 'active' => true]
        ];

        $currencies = Currency::cases();

        $tradeBot = TradeBot::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Edit Trade Bot',
            'breadcrumbs' => $breadcrumbs,
            'currencies' => $currencies,
            'tradeBot' => $tradeBot
        ];

        return view('dashboard.admin.trade_bot.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateTradeBotRequest $request, string $uuid)
    {
        try {
            $tradeBot = TradeBot::where('uuid', $uuid)->firstOrFail();

            DB::transaction(function () use ($request, $tradeBot) {
                $tradeBot->update($request->validated());
            });

            return redirect()
                ->route('admin.trade.bot.index')
                ->with('success', 'Trade Bot updated successfully');
        } catch (\Exception $e) {
            Log::error($e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'Failed to update trade bot');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function delete(string $uuid)
    {
        try {
            $tradeBot = TradeBot::where('uuid', $uuid)->firstOrFail();

            DB::transaction(function () use ($tradeBot) {
                $tradeBot->delete();
            });

            return redirect()
                ->route('admin.trade.bot.index')
                ->with('success', 'Trade Bot deleted successfully');
        } catch (\Exception $e) {
            Log::error($e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'Failed to delete trade bot');
        }
    }
}
