<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\User;
use App\Models\Transaction;
use App\Enum\UserPlanStatus;
use Illuminate\Http\Request;
use App\Enum\TransactionStatus;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;

class UserPlanController extends Controller
{
    public function index(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Users', 'url' => route('admin.user.index')],
            ['label' => 'User Details', 'url' => route('admin.user.show', $uuid)],
            ['label' => 'Subscription Plans', 'active' => true],
        ];

        $user = User::where('uuid', $uuid)->firstOrFail();
        $userPlans = $user->userPlans()->latest()->get();

        $data = [
            'title' => 'Subscription Plans',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'userPlans' => $userPlans
        ];

        return view('dashboard.admin.user.plan.index', $data);
    }

    public function show(string $userUUID, string $userPlanUUID)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Users', 'url' => route('admin.user.index')],
            ['label' => 'User Details', 'url' => route('admin.user.show', $userUUID)],
            ['label' => 'Subscription Plans', 'url' => route('admin.user.plan.index', $userUUID)],
            ['label' => 'Plan Details', 'active' => true],
        ];

        $user = User::where('uuid', $userUUID)->firstOrFail();
        $userPlan = $user->userPlans()->where('uuid', $userPlanUUID)->firstOrFail();
        $transaction = Transaction::where('other_reference_id', $userPlan->uuid)->firstOrFail();

        $userPlanStatues = UserPlanStatus::cases();

        $data = [
            'title' => 'Plan Details',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'userPlan' => $userPlan,
            'transaction' => $transaction,
            'userPlanStatues' => $userPlanStatues
        ];

        return view('dashboard.admin.user.plan.show', $data);
    }

    public function update(Request $request, string $userUUID, string $UserPlanUUID)
    {
        $request->validate([
            'status' => ['required', 'in:' . implode(',', array_column(UserPlanStatus::cases(), 'value'))]
        ]);

        try {

            $user = User::where('uuid', $userUUID)->firstOrFail();
            $userPlan = $user->userPlans()->where('uuid', $UserPlanUUID)->firstOrFail();
            $transaction = Transaction::where('other_reference_id', $userPlan->uuid)->firstOrFail();

            DB::beginTransaction();

            switch ($request->status) {
                case 'completed':
                    $userPlan->update([
                        'starts_at' => now(),
                        'ends_at' => now()->addDays($userPlan->duration),
                        'status' => UserPlanStatus::COMPLETED->value
                    ]);

                    $user->balance -= $userPlan->amount;
                    $user->save();

                    $transaction->update([
                        'status' => TransactionStatus::COMPLETED->value
                    ]);


                    break;

                case 'cancelled':
                    $userPlan->update([
                        'status' => UserPlanStatus::CANCELLED->value
                    ]);

                    $transaction->update([
                        'status' => TransactionStatus::CANCELLED->value
                    ]);

                    break;

                case 'pending':
                    $userPlan->update([
                        'status' => UserPlanStatus::PENDING->value
                    ]);

                    $transaction->update([
                        'status' => TransactionStatus::PENDING->value
                    ]);

                    break;

                default:
                    $userPlan->update([
                        'status' => $request->status
                    ]);
                    break;
            }


            DB::commit();

            return redirect()->back()->with('success', 'Plan status updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
}
