<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\User;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Enum\TransactionType;
use App\Enum\TransactionStatus;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;

class UserWithdrawController extends Controller
{
    public function index(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Users', 'url' => route('admin.user.index')],
            ['label' => 'User Details', 'url' => route('admin.user.show', $uuid)],
            ['label' => 'Withdraws', 'active' => true],
        ];

        $user = User::where('uuid', $uuid)->firstOrFail();

        $transactions = Transaction::where('user_id', $user->id)->where('type', TransactionType::WITHDRAW)->latest()->get();

        $data = [
            'title' => 'Withdraws',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'transactions' => $transactions
        ];

        return view('dashboard.admin.user.withdraw.index', $data);
    }

    public function show(string $userUUID, string $transactionUUID)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Users', 'url' => route('admin.user.index')],
            ['label' => 'User Details', 'url' => route('admin.user.show', $userUUID)],
            ['label' => 'Withdraws', 'url' => route('admin.user.withdraw.index', $userUUID)],
            ['label' => 'Withdraw Details', 'active' => true],
        ];

        $user = User::where('uuid', $userUUID)->firstOrFail();
        $transaction = $user->transaction()->where('uuid', $transactionUUID)->firstOrFail();
        $transactionStatues = TransactionStatus::cases();

        $data = [
            'title' => 'Withdraw Details',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'transaction' => $transaction,
            'transactionStatues' => $transactionStatues
        ];

        return view('dashboard.admin.user.withdraw.show', $data);
    }

    public function update(Request $request, string $userUUID, string $transactionUUID)
    {
        $request->validate([
            'status' => ['required', 'in:' . implode(',', array_column(TransactionStatus::cases(), 'value'))]
        ]);

        try {

            $user = User::where('uuid', $userUUID)->firstOrFail();
            $transaction = $user->transaction()->where('uuid', $transactionUUID)->firstOrFail();

            DB::beginTransaction();

            switch ($request->status) {
                case 'completed':
                    $transaction->update([
                        'status' => TransactionStatus::COMPLETED->value
                    ]);

                    $user->balance -= $transaction->amount;
                    $user->save();

                    break;

                default:
                    $transaction->update([
                        'status' => $request->status
                    ]);
                    break;
            }


            DB::commit();

            return redirect()->back()->with('success', 'Transaction status updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    public function delete(string $userUUID, string $transactionUUID)
    {
        try {
            $user = User::where('uuid', $userUUID)->firstOrFail();
            $transaction = $user->transaction()->where('uuid', $transactionUUID)->firstOrFail();

            DB::beginTransaction();

            $transaction->delete();

            DB::commit();

            return redirect()->route('admin.user.withdraw.index', $userUUID)->with('success', 'Withdraw transaction deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }
}
