<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\Wallet;
use App\Enum\WalletStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreWalletRequest;
use App\Http\Requests\UpdateWalletRequest;
use App\Trait\FileUpload;

class WalletController extends Controller
{
    use FileUpload;

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Wallets', 'active' => true]
        ];

        $data = [
            'title' => 'Wallets',
            'breadcrumbs' => $breadcrumbs,
            'wallets' => Wallet::latest()->get()
        ];

        return view('dashboard.admin.wallet.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Wallets', 'url' => route('admin.wallet.index')],
            ['label' => 'Create Wallet', 'active' => true]
        ];

        $walletStatues = WalletStatus::cases();

        $data = [
            'title' => 'Create Wallet',
            'breadcrumbs' => $breadcrumbs,
            'walletStatues' => $walletStatues
        ];

        return view('dashboard.admin.wallet.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreWalletRequest $request)
    {
        try {
            DB::beginTransaction();

            $wallet = Wallet::create([
                'uuid' => str()->uuid(),
                'name' => ucwords($request->name),
                'symbol' => $request->symbol,
                'address' => $request->address,
                'network' => $request->network,
                'qr_code_path' => $this->uploadFile($request, 'qr_code_path', uploadPathAdmin('wallet')),
                'logo_path' => $this->imageInterventionUploadFile($request, 'logo_path', uploadPathAdmin('wallet'), 48, 48),
                'balance' => $request->balance,
                'status' => $request->status,
                'description' => $request->description,
            ]);

            DB::commit();
            return redirect()->route('admin.wallet.show', $wallet->uuid)->with('success', 'Wallet created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Wallet $wallet)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Wallets', 'url' => route('admin.wallet.index')],
            ['label' => 'Wallet Details', 'active' => true]
        ];

        $data = [
            'title' => 'Wallet Details',
            'breadcrumbs' => $breadcrumbs,
            'wallet' => $wallet
        ];

        return view('dashboard.admin.wallet.show', $data);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Wallet $wallet)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Wallets', 'url' => route('admin.wallet.index')],
            ['label' => 'Edit Wallet', 'active' => true]
        ];

        $walletStatues = WalletStatus::cases();

        $data = [
            'title' => 'Edit Wallet',
            'breadcrumbs' => $breadcrumbs,
            'walletStatues' => $walletStatues,
            'wallet' => $wallet
        ];

        return view('dashboard.admin.wallet.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateWalletRequest $request, Wallet $wallet)
    {
        try {
            DB::beginTransaction();

            $wallet->update([
                'name' => $request->name,
                'symbol' => $request->symbol,
                'address' => $request->address,
                'network' => $request->network,
                'qr_code_path' => $this->updateFile($request, 'qr_code_path', uploadPathAdmin('wallet'), $wallet?->qr_code_path),
                'logo_path' => $this->imageInterventionUpdateFile($request, 'logo_path', uploadPathAdmin('wallet'), 48, 48, $wallet?->logo_path),
                'balance' => $request->balance,
                'status' => $request->status,
                'description' => $request->description,
            ]);

            DB::commit();
            return redirect()->route('admin.wallet.show', $wallet->uuid)->with('success', 'Wallet updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function delete(Wallet $wallet)
    {
        try {
            DB::beginTransaction();

            $walletQrCode = $wallet->qr_code_path;

            $walletLogo = $wallet->logo_path;

            $wallet->delete();

            $this->deleteFile($walletQrCode);

            $this->deleteFile($walletLogo);

            DB::commit();

            return redirect()->route('admin.wallet.index')->with('success', 'Wallet deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }
}
