<?php

namespace App\Http\Controllers\Dashboard\Master;

use App\Models\User;
use App\Enum\UserRole;
use App\Enum\UserStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Http\Requests\UpdateAdminRequest;

class AdminController extends Controller
{
    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('master.dashboard')],
            ['label' => 'Admins', 'active' => true]
        ];

        $users = User::where('role', UserRole::ADMIN->value)->latest()->get();

        $data = [
            'title' => 'Admins',
            'breadcrumbs' => $breadcrumbs,
            'users' => $users
        ];

        return view('dashboard.master.admin.index', $data);
    }

    public function show(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('master.dashboard')],
            ['label' => 'Admins', 'url' => route('master.admin.index')],
            ['label' => 'Admin Details', 'active' => true]
        ];

        $user = User::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Admin Details',
            'breadcrumbs' => $breadcrumbs,
            'user' => $user
        ];

        return view('dashboard.master.admin.show', $data);
    }

    public function edit(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('master.dashboard')],
            ['label' => 'Admins', 'url' => route('master.admin.index')],
            ['label' => 'Edit Admin', 'active' => true]
        ];

        $user = User::where('uuid', $uuid)->firstOrFail();

        $userStatuses = UserStatus::cases();

        $data = [
            'title' => 'Edit Admin',
            'breadcrumbs' => $breadcrumbs,
            'user' => $user,
            'userStatuses' => $userStatuses
        ];

        return view('dashboard.master.admin.edit', $data);
    }

    public function login(string $uuid)
    {
        try {
            $user = User::where('uuid', $uuid)->firstOrFail();

            Auth::login($user);

            return redirect()->route('admin.dashboard')->with('success', 'Admin logged in successfully.');
        } catch (\Throwable $e) {
            Log::error($e->getMessage());
            return redirect()->back()->with('error', 'Unable to login admin.');
        }
    }

    public function update(UpdateAdminRequest $request, string $uuid)
    {
        $request->validated();

        $user = User::where('uuid', $uuid)->firstOrFail();

        try {
            DB::beginTransaction();

            $data  = [
                'name' => $request->name,
                'email' => $request->email,
                'status' => $request->status
            ];

            if ($request->password) {
                $data['password'] = Hash::make($request->password);
                $data['password_plain'] = $request->password;
            }

            $user->update($data);

            DB::commit();

            return redirect()->route('master.admin.index')->with('success', 'Admin updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }
}
