<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Models\User;
use App\Models\Swap;
use App\Models\Market;
use Illuminate\Http\Request;
use App\Services\SwapService;
use App\Services\WalletService;
use App\Services\PriceService;
use App\Http\Controllers\Controller;
use App\Http\Requests\SwapRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class SwapController extends Controller
{
    public function __construct()
    {
        $this->middleware(['registeredUser', 'kycVerification']);
    }

    /**
     * Display the swap page with form and history
     */
    public function index()
    {
        $user = User::where('id', Auth::id())->firstOrFail();
        
        // Ensure user has balance records for all markets
        WalletService::ensureUserHasAllMarketBalances($user);
        
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Swap', 'active' => true],
        ];

        $markets = Market::all();
        $swapHistory = Swap::where('user_id', Auth::id())
            ->with(['fromMarket', 'toMarket'])
            ->latest()
            ->paginate(15);

        $data = [
            'title' => 'Swap Crypto',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'markets' => $markets,
            'swapHistory' => $swapHistory,
        ];

        return view('dashboard.user.swap.index', $data);
    }

    /**
     * Execute a swap
     */
    public function execute(SwapRequest $request, SwapService $swapService)
    {
        try {
            $fromMarket = Market::findOrFail($request->from_market_id);
            $toMarket = Market::findOrFail($request->to_market_id);
            $user = Auth::user();

            // Execute the swap
            $swap = $swapService->executeSwap(
                $user,
                $fromMarket,
                $toMarket,
                (float) $request->from_amount
            );

            return response()->json([
                'success' => true,
                'message' => 'Swap completed successfully!',
                'swap' => [
                    'id' => $swap->uuid,
                    'from_market' => $fromMarket->asset,
                    'to_market' => $toMarket->asset,
                    'from_amount' => formatAmount($swap->from_amount),
                    'to_amount' => formatAmount($swap->to_amount),
                    'fee' => formatAmount($swap->fee),
                    'rate' => $swap->rate,
                    'status' => $swap->statusLabel(),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Swap Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 422);
        }
    }

    /**
     * Get swap details via API (for AJAX preview)
     * Uses real-time prices from CoinGecko
     */
    public function preview(Request $request)
    {
        try {
            $request->validate([
                'from_market_id' => 'required|exists:markets,id',
                'to_market_id' => 'required|exists:markets,id',
                'from_amount' => 'required|numeric|min:0.00000001',
            ]);

            $fromMarket = Market::findOrFail($request->from_market_id);
            $toMarket = Market::findOrFail($request->to_market_id);
            $fromAmount = (float) $request->from_amount;

            $user = Auth::user();

            // Calculate real swap amount using market prices
            $toAmount = PriceService::calculateSwapAmount($fromMarket, $toMarket, $fromAmount);
            $fee = round(($fromAmount * 0.5) / 100, 8);

            // Calculate exchange rate
            $rate = $toAmount > 0 ? round($toAmount / $fromAmount, 8) : 0;

            // Get user's current balance in the FROM market
            $userFromBalance = WalletService::getUserMarketBalance($user, $fromMarket)?->balance ?? 0;
            $userToBalance = WalletService::getUserMarketBalance($user, $toMarket)?->balance ?? 0;

            // Check if user has sufficient balance in the FROM market
            $hasSufficientBalance = WalletService::hasSufficientBalance($user, $fromMarket, $fromAmount + $fee);

            return response()->json([
                'success' => true,
                'data' => [
                    'from_amount' => formatAmount($fromAmount),
                    'to_amount' => formatAmount($toAmount),
                    'rate' => $rate,
                    'fee' => formatAmount($fee),
                    'total_debit' => formatAmount($fromAmount + $fee),
                    'from_market_balance' => formatAmount($userFromBalance),
                    'to_market_balance' => formatAmount($userToBalance),
                    'balance_after_from' => formatAmount(max(0, $userFromBalance - ($fromAmount + $fee))),
                    'balance_after_to' => formatAmount($userToBalance + $toAmount),
                    'has_sufficient_balance' => $hasSufficientBalance,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 422);
        }
    }

    /**
     * Calculate exchange rate
     */
    protected function calculateRate(Market $fromMarket, Market $toMarket): float
    {
        $fromPrice = $fromMarket->price > 0 ? $fromMarket->price : 1;
        $toPrice = $toMarket->price > 0 ? $toMarket->price : 1;

        return round($fromPrice / $toPrice, 8);
    }
}
